<?php
/**
 * WireGuard Telegram Bot - Webhook Handler
 * هندلر وب‌هوک ربات تلگرام وایرگارد
 * 
 * نصب:
 * 1. فایل‌ها را روی سرور آپلود کنید
 * 2. config.php را ویرایش کنید
 * 3. وب‌هوک را تنظیم کنید: https://yourdomain.com/webhook.php?action=setwebhook
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Load dependencies
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/TelegramBot.php';
require_once __DIR__ . '/WireGuardManager.php';
require_once __DIR__ . '/MikroTikAPI.php';

$config = require __DIR__ . '/config.php';

// Initialize components
$bot = new TelegramBot($config['telegram']['bot_token'], $config['telegram']['admin_ids']);
$wgManager = new WireGuardManager($config);

// Handle webhook setup
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['action']) {
        case 'setwebhook':
            $webhookUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") 
                . "://{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";
            $webhookUrl = preg_replace('/\?.*$/', '', $webhookUrl);
            
            $result = $bot->setWebhook($webhookUrl);
            echo json_encode(['webhook_url' => $webhookUrl, 'result' => $result], JSON_PRETTY_PRINT);
            exit;
            
        case 'deletewebhook':
            $result = $bot->deleteWebhook();
            echo json_encode($result, JSON_PRETTY_PRINT);
            exit;
            
        case 'webhookinfo':
            $result = $bot->getWebhookInfo();
            echo json_encode($result, JSON_PRETTY_PRINT);
            exit;
            
        case 'stats':
            $stats = $wgManager->getStats();
            echo json_encode($stats, JSON_PRETTY_PRINT);
            exit;
    }
}

// Get update from Telegram
$update = $bot->getUpdate();

if (!$update) {
    exit;
}

// Log update for debugging
file_put_contents(__DIR__ . '/data/log.txt', date('Y-m-d H:i:s') . " - " . json_encode($update) . "\n", FILE_APPEND);

// Process update
try {
    if (isset($update['message'])) {
        handleMessage($update['message'], $bot, $wgManager, $config);
    } elseif (isset($update['callback_query'])) {
        handleCallbackQuery($update['callback_query'], $bot, $wgManager, $config);
    }
} catch (Exception $e) {
    error_log("Bot Error: " . $e->getMessage());
    if (isset($update['message']['chat']['id'])) {
        $bot->sendMessage($update['message']['chat']['id'], "❌ خطا: " . $e->getMessage());
    }
}

/**
 * Handle incoming messages
 */
function handleMessage(array $message, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $text = $message['text'] ?? '';
    $username = $message['from']['username'] ?? 'unknown';
    
    // Check if admin
    if (!$bot->isAdmin($userId)) {
        $bot->sendMessage($chatId, "⛔ شما دسترسی به این ربات ندارید.\n\nآیدی شما: <code>$userId</code>");
        return;
    }
    
    // Handle commands
    if (strpos($text, '/') === 0) {
        $parts = explode(' ', $text, 2);
        $command = strtolower($parts[0]);
        $args = $parts[1] ?? '';
        
        switch ($command) {
            case '/start':
                sendWelcome($chatId, $bot);
                break;
                
            case '/help':
                sendHelp($chatId, $bot);
                break;
                
            case '/new':
            case '/add':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/new user1</code>");
                    return;
                }
                createNewPeer($chatId, $userId, $args, $bot, $wgManager, $config);
                break;
                
            case '/list':
                listPeers($chatId, $bot, $wgManager);
                break;
                
            case '/delete':
            case '/remove':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/delete user1</code>");
                    return;
                }
                deletePeer($chatId, $args, $bot, $wgManager, $config);
                break;
                
            case '/get':
            case '/config':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/get user1</code>");
                    return;
                }
                getPeerConfig($chatId, $args, $bot, $wgManager);
                break;
                
            case '/stats':
                showStats($chatId, $bot, $wgManager);
                break;
                
            case '/sync':
                syncWithMikroTik($chatId, $bot, $wgManager, $config);
                break;

            case '/usage':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/usage user1</code>");
                    return;
                }
                showPeerUsage($chatId, $args, $bot, $wgManager, $config);
                break;

            case '/enable':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/enable user1</code>");
                    return;
                }
                togglePeer($chatId, $args, true, $bot, $wgManager, $config);
                break;

            case '/disable':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/disable user1</code>");
                    return;
                }
                togglePeer($chatId, $args, false, $bot, $wgManager, $config);
                break;

            case '/info':
                if (empty($args)) {
                    $bot->sendMessage($chatId, "❌ لطفاً نام peer را وارد کنید.\n\nمثال: <code>/info user1</code>");
                    return;
                }
                showPeerInfo($chatId, $args, $bot, $wgManager, $config);
                break;

            default:
                $bot->sendMessage($chatId, "❓ دستور ناشناخته. برای راهنما /help را بزنید.");
        }
    } else {
        // اگر فقط یک نام ارسال شد، آن را به عنوان peer جدید در نظر بگیر
        $name = trim($text);
        if (preg_match('/^[a-zA-Z0-9_-]+$/', $name)) {
            // تأیید از کاربر
            $bot->sendInlineKeyboard($chatId, 
                "🔑 آیا می‌خواهید peer جدید با نام <b>$name</b> ایجاد کنید?",
                [
                    [
                        ['text' => '✅ بله، ایجاد کن', 'callback_data' => "create:$name"],
                        ['text' => '❌ خیر', 'callback_data' => 'cancel'],
                    ]
                ]
            );
        } else {
            $bot->sendMessage($chatId, "❓ برای ایجاد peer جدید، نام معتبر (فقط حروف انگلیسی، اعداد، _ و -) وارد کنید.\n\nیا از /help برای راهنما استفاده کنید.");
        }
    }
}

/**
 * Handle callback queries
 */
function handleCallbackQuery(array $callback, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $callbackId = $callback['id'];
    $chatId = $callback['message']['chat']['id'];
    $userId = $callback['from']['id'];
    $data = $callback['data'];
    
    if (!$bot->isAdmin($userId)) {
        $bot->answerCallbackQuery($callbackId, '⛔ دسترسی ندارید', true);
        return;
    }
    
    if (strpos($data, 'create:') === 0) {
        $name = substr($data, 7);
        $bot->answerCallbackQuery($callbackId, 'در حال ایجاد...');
        createNewPeer($chatId, $userId, $name, $bot, $wgManager, $config);
    } elseif (strpos($data, 'delete_confirm:') === 0) {
        $name = substr($data, 15);
        $bot->answerCallbackQuery($callbackId, 'در حال حذف...');
        performDelete($chatId, $name, $bot, $wgManager, $config);
    } elseif ($data === 'cancel') {
        $bot->answerCallbackQuery($callbackId, 'لغو شد');
        $bot->editMessageText($chatId, $callback['message']['message_id'], '❌ عملیات لغو شد.');
    }
}

/**
 * Send welcome message
 */
function sendWelcome(int $chatId, TelegramBot $bot): void
{
    $text = "👋 <b>به ربات مدیریت WireGuard خوش آمدید!</b>\n\n";
    $text .= "این ربات به شما امکان می‌دهد:\n";
    $text .= "• ایجاد کانفیگ WireGuard جدید\n";
    $text .= "• دریافت QR Code و فایل conf\n";
    $text .= "• اضافه کردن خودکار به میکروتیک\n\n";
    $text .= "🚀 برای شروع، فقط یک نام تایپ کنید یا /help را بزنید.";
    
    $bot->sendMessage($chatId, $text);
}

/**
 * Send help message
 */
function sendHelp(int $chatId, TelegramBot $bot): void
{
    $text = "📖 <b>راهنمای دستورات:</b>\n\n";
    $text .= "<b>مدیریت Peer:</b>\n";
    $text .= "/new [نام] - ایجاد peer جدید با سرعت 20M/20M\n";
    $text .= "/list - لیست همه peerها\n";
    $text .= "/get [نام] - دریافت کانفیگ یک peer\n";
    $text .= "/delete [نام] - حذف یک peer\n";
    $text .= "/enable [نام] - فعال کردن یک peer\n";
    $text .= "/disable [نام] - غیرفعال کردن یک peer\n\n";
    $text .= "<b>اطلاعات و آمار:</b>\n";
    $text .= "/info [نام] - اطلاعات کامل یک peer\n";
    $text .= "/usage [نام] - میزان مصرف یک peer\n";
    $text .= "/stats - آمار کلی سیستم\n";
    $text .= "/sync - همگام‌سازی با میکروتیک\n\n";
    $text .= "💡 <b>نکته:</b> می‌توانید فقط یک نام تایپ کنید تا peer جدید ایجاد شود.";

    $bot->sendMessage($chatId, $text);
}

/**
 * Create new peer
 */
function createNewPeer(int $chatId, int $userId, string $name, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $name = trim(preg_replace('/[^a-zA-Z0-9_-]/', '', $name));
    
    if (empty($name)) {
        $bot->sendMessage($chatId, "❌ نام نامعتبر است. فقط از حروف انگلیسی، اعداد، _ و - استفاده کنید.");
        return;
    }
    
    // Check if exists
    if ($wgManager->getPeerByName($name)) {
        $bot->sendMessage($chatId, "❌ یک peer با نام <b>$name</b> قبلاً وجود دارد.");
        return;
    }
    
    $bot->sendMessage($chatId, "⏳ در حال ایجاد peer...");

    try {
        // First, create peer in database to get keys and IP
        $peer = $wgManager->createPeer($name, $userId);

        // Try to add to MikroTik FIRST - if it fails, delete peer and stop
        $mikrotikResult = addToMikroTik($peer, $wgManager, $config);

        if (!$mikrotikResult['success']) {
            // MikroTik failed - delete peer from database and stop
            $wgManager->deletePeer($peer['id']);
            $bot->sendMessage($chatId, "❌ خطا در اتصال به میکروتیک:\n<code>" . $mikrotikResult['error'] . "</code>\n\nPeer ایجاد نشد. لطفاً پس از رفع مشکل دوباره تلاش کنید.");
            return;
        }

        // MikroTik succeeded - now generate config and QR
        $configFile = $wgManager->saveConfigFile($peer);
        $qrFile = $wgManager->generateQRCode($peer);

        // Send results
        $text = "✅ <b>Peer جدید ایجاد شد!</b>\n\n";
        $text .= "📛 <b>نام:</b> <code>{$peer['name']}</code>\n";
        $text .= "🌐 <b>IP:</b> <code>{$peer['ip']}</code>\n";
        $text .= "🔑 <b>Public Key:</b>\n<code>{$peer['public_key']}</code>\n\n";
        $text .= "✅ به میکروتیک اضافه شد";

        $bot->sendMessage($chatId, $text);

        // Send QR code
        $bot->sendPhoto($chatId, $qrFile, "📱 QR Code برای <b>{$peer['name']}</b>");

        // Send config file
        $bot->sendDocument($chatId, $configFile, "📄 فایل کانفیگ برای <b>{$peer['name']}</b>");

    } catch (Exception $e) {
        $bot->sendMessage($chatId, "❌ خطا در ایجاد peer: " . $e->getMessage());
    }
}

/**
 * Add peer to MikroTik with 20M/20M rate limit
 */
function addToMikroTik(array $peer, WireGuardManager $wgManager, array $config): array
{
    try {
        $mt = new MikroTikAPI(
            $config['mikrotik']['host'],
            $config['mikrotik']['port'],
            $config['mikrotik']['username'],
            $config['mikrotik']['password']
        );

        $mt->connect();

        // Add peer with 20M/20M rate limit
        $result = $mt->addWireGuardPeer(
            $config['wireguard']['interface'],
            $peer['public_key'],
            $peer['ip'] . '/32',
            $peer['name'],
            '20M',  // rx-rate: download speed for client (server receives upload from client)
            '20M'   // tx-rate: upload speed for client (server transmits download to client)
        );

        $mt->disconnect();

        // Check result - handle MikroTik API response format
        // Success: ['!done'] or ['.id' => 'xxx', '!done']
        // Error: [['message' => 'error text'], '!trap', '!done']
        foreach ($result as $item) {
            if ($item === '!done') {
                $wgManager->markMikroTikAdded($peer['id']);
                return ['success' => true];
            }
            if ($item === '!trap') {
                // Find error message in response
                foreach ($result as $r) {
                    if (is_array($r) && isset($r['message'])) {
                        return ['success' => false, 'error' => 'MikroTik: ' . $r['message']];
                    }
                }
                return ['success' => false, 'error' => 'MikroTik returned an error'];
            }
            if ($item === '!fatal') {
                return ['success' => false, 'error' => 'MikroTik fatal error'];
            }
        }

        return ['success' => false, 'error' => 'Unknown response from MikroTik: ' . json_encode($result)];

    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Remove peer from MikroTik
 */
function removeFromMikroTik(array $peer, array $config): array
{
    try {
        $mt = new MikroTikAPI(
            $config['mikrotik']['host'],
            $config['mikrotik']['port'],
            $config['mikrotik']['username'],
            $config['mikrotik']['password']
        );
        
        $mt->connect();
        
        $existingPeer = $mt->findPeerByPublicKey($peer['public_key']);
        
        if ($existingPeer && isset($existingPeer['.id'])) {
            $result = $mt->removeWireGuardPeer($existingPeer['.id']);
            $mt->disconnect();

            // Check for success or error in response
            foreach ($result as $item) {
                if ($item === '!done') {
                    return ['success' => true];
                }
                if ($item === '!trap') {
                    foreach ($result as $r) {
                        if (is_array($r) && isset($r['message'])) {
                            return ['success' => false, 'error' => 'MikroTik: ' . $r['message']];
                        }
                    }
                    return ['success' => false, 'error' => 'MikroTik returned an error'];
                }
            }
            return ['success' => false, 'error' => 'Unknown response from MikroTik: ' . json_encode($result)];
        }

        $mt->disconnect();
        return ['success' => false, 'error' => 'Peer not found in MikroTik'];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * List all peers
 */
function listPeers(int $chatId, TelegramBot $bot, WireGuardManager $wgManager): void
{
    $peers = $wgManager->getAllPeers();
    
    if (empty($peers)) {
        $bot->sendMessage($chatId, "📭 هیچ peer‌ای وجود ندارد.");
        return;
    }
    
    $text = "📋 <b>لیست Peerها:</b>\n\n";
    
    foreach ($peers as $peer) {
        $mtStatus = $peer['mikrotik_added'] ? '✅' : '❌';
        $text .= "• <b>{$peer['name']}</b> - {$peer['ip']} $mtStatus\n";
    }
    
    $text .= "\n✅ = در میکروتیک\n❌ = فقط لوکال";
    
    $bot->sendMessage($chatId, $text);
}

/**
 * Delete peer - ask for confirmation
 */
function deletePeer(int $chatId, string $name, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $peer = $wgManager->getPeerByName($name);
    
    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer با نام <b>$name</b> یافت نشد.");
        return;
    }
    
    $bot->sendInlineKeyboard($chatId,
        "⚠️ آیا مطمئن هستید که می‌خواهید peer <b>{$peer['name']}</b> را حذف کنید?\n\nIP: {$peer['ip']}",
        [
            [
                ['text' => '🗑 بله، حذف کن', 'callback_data' => "delete_confirm:{$peer['name']}"],
                ['text' => '❌ خیر', 'callback_data' => 'cancel'],
            ]
        ]
    );
}

/**
 * Perform delete
 */
function performDelete(int $chatId, string $name, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $peer = $wgManager->getPeerByName($name);
    
    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer یافت نشد.");
        return;
    }
    
    // Remove from MikroTik
    if ($peer['mikrotik_added']) {
        $mtResult = removeFromMikroTik($peer, $config);
    }
    
    // Delete locally
    $wgManager->deletePeer($peer['id']);
    
    $text = "🗑 <b>Peer حذف شد!</b>\n\n";
    $text .= "📛 نام: {$peer['name']}\n";
    $text .= "🌐 IP: {$peer['ip']}\n";
    
    if (isset($mtResult)) {
        if ($mtResult['success']) {
            $text .= "\n✅ از میکروتیک حذف شد";
        } else {
            $text .= "\n⚠️ خطا در حذف از میکروتیک: " . $mtResult['error'];
        }
    }
    
    $bot->sendMessage($chatId, $text);
}

/**
 * Get peer config
 */
function getPeerConfig(int $chatId, string $name, TelegramBot $bot, WireGuardManager $wgManager): void
{
    $peer = $wgManager->getPeerByName($name);
    
    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer با نام <b>$name</b> یافت نشد.");
        return;
    }
    
    // Generate files if not exist
    $configFile = $wgManager->saveConfigFile($peer);
    $qrFile = $wgManager->generateQRCode($peer);
    
    $text = "📋 <b>اطلاعات Peer:</b>\n\n";
    $text .= "📛 <b>نام:</b> <code>{$peer['name']}</code>\n";
    $text .= "🌐 <b>IP:</b> <code>{$peer['ip']}</code>\n";
    $text .= "🔑 <b>Public Key:</b>\n<code>{$peer['public_key']}</code>\n";
    $text .= "📅 <b>تاریخ ایجاد:</b> {$peer['created_at']}\n";
    $text .= "🔄 <b>در میکروتیک:</b> " . ($peer['mikrotik_added'] ? '✅' : '❌');
    
    $bot->sendMessage($chatId, $text);
    $bot->sendPhoto($chatId, $qrFile, "📱 QR Code");
    $bot->sendDocument($chatId, $configFile, "📄 فایل کانفیگ");
}

/**
 * Show stats
 */
function showStats(int $chatId, TelegramBot $bot, WireGuardManager $wgManager): void
{
    $stats = $wgManager->getStats();
    
    $text = "📊 <b>آمار سیستم:</b>\n\n";
    $text .= "👥 تعداد کل Peerها: {$stats['total_peers']}\n";
    $text .= "🌐 IP‌های استفاده شده: {$stats['used_ips']}\n";
    $text .= "✅ IP‌های آزاد: {$stats['available_ips']}\n";
    $text .= "📦 ظرفیت کل: {$stats['total_ips']}";
    
    $bot->sendMessage($chatId, $text);
}

/**
 * Sync with MikroTik
 */
function syncWithMikroTik(int $chatId, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $bot->sendMessage($chatId, "⏳ در حال همگام‌سازی با میکروتیک...");

    try {
        $mt = new MikroTikAPI(
            $config['mikrotik']['host'],
            $config['mikrotik']['port'],
            $config['mikrotik']['username'],
            $config['mikrotik']['password']
        );

        $mt->connect();
        $mtPeers = $mt->getWireGuardPeers($config['wireguard']['interface']);
        $mt->disconnect();

        $localPeers = $wgManager->getAllPeers();

        $text = "📊 <b>نتیجه همگام‌سازی:</b>\n\n";
        $text .= "🔹 Peerهای میکروتیک: " . count($mtPeers) . "\n";
        $text .= "🔹 Peerهای لوکال: " . count($localPeers) . "\n\n";

        // Check each local peer
        $synced = 0;
        $notSynced = 0;

        foreach ($localPeers as $peer) {
            $found = false;
            foreach ($mtPeers as $mtPeer) {
                if (isset($mtPeer['public-key']) && $mtPeer['public-key'] === $peer['public_key']) {
                    $found = true;
                    break;
                }
            }

            if ($found) {
                $synced++;
                $wgManager->markMikroTikAdded($peer['id']);
            } else {
                $notSynced++;
            }
        }

        $text .= "✅ همگام: $synced\n";
        $text .= "❌ ناهمگام: $notSynced";

        $bot->sendMessage($chatId, $text);

    } catch (Exception $e) {
        $bot->sendMessage($chatId, "❌ خطا در ارتباط با میکروتیک: " . $e->getMessage());
    }
}

/**
 * Show peer usage statistics
 */
function showPeerUsage(int $chatId, string $name, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $peer = $wgManager->getPeerByName($name);

    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer با نام <b>$name</b> یافت نشد.");
        return;
    }

    if (!$peer['mikrotik_added']) {
        $bot->sendMessage($chatId, "⚠️ این peer به میکروتیک اضافه نشده است.\n\nابتدا با دستور /sync همگام‌سازی کنید.");
        return;
    }

    try {
        $mt = new MikroTikAPI(
            $config['mikrotik']['host'],
            $config['mikrotik']['port'],
            $config['mikrotik']['username'],
            $config['mikrotik']['password']
        );

        $mt->connect();
        $mtPeer = $mt->findPeerByPublicKey($peer['public_key']);

        if (!$mtPeer || !isset($mtPeer['.id'])) {
            $mt->disconnect();
            $bot->sendMessage($chatId, "❌ Peer در میکروتیک یافت نشد.");
            return;
        }

        $traffic = $mt->getPeerTraffic($mtPeer['.id']);
        $mt->disconnect();

        if (!$traffic) {
            $bot->sendMessage($chatId, "❌ خطا در دریافت اطلاعات ترافیک.");
            return;
        }

        // Format bytes to human readable
        $rxFormatted = formatBytes($traffic['rx_bytes']);
        $txFormatted = formatBytes($traffic['tx_bytes']);
        $totalFormatted = formatBytes($traffic['rx_bytes'] + $traffic['tx_bytes']);

        $text = "📊 <b>آمار مصرف: {$peer['name']}</b>\n\n";
        $text .= "📥 <b>دانلود (از دید کلاینت):</b> $txFormatted\n";
        $text .= "📤 <b>آپلود (از دید کلاینت):</b> $rxFormatted\n";
        $text .= "📦 <b>مجموع:</b> $totalFormatted\n\n";

        if ($traffic['last_handshake']) {
            $text .= "🤝 <b>آخرین handshake:</b> {$traffic['last_handshake']}\n";
        }

        if ($traffic['current_endpoint_address']) {
            $text .= "🌐 <b>IP کلاینت:</b> {$traffic['current_endpoint_address']}:{$traffic['current_endpoint_port']}\n";
        }

        $status = $traffic['disabled'] ? '❌ غیرفعال' : '✅ فعال';
        $text .= "🔄 <b>وضعیت:</b> $status";

        $bot->sendMessage($chatId, $text);

    } catch (Exception $e) {
        $bot->sendMessage($chatId, "❌ خطا در ارتباط با میکروتیک: " . $e->getMessage());
    }
}

/**
 * Toggle peer enable/disable
 */
function togglePeer(int $chatId, string $name, bool $enable, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $peer = $wgManager->getPeerByName($name);

    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer با نام <b>$name</b> یافت نشد.");
        return;
    }

    if (!$peer['mikrotik_added']) {
        $bot->sendMessage($chatId, "⚠️ این peer به میکروتیک اضافه نشده است.\n\nابتدا با دستور /sync همگام‌سازی کنید.");
        return;
    }

    try {
        $mt = new MikroTikAPI(
            $config['mikrotik']['host'],
            $config['mikrotik']['port'],
            $config['mikrotik']['username'],
            $config['mikrotik']['password']
        );

        $mt->connect();
        $mtPeer = $mt->findPeerByPublicKey($peer['public_key']);

        if (!$mtPeer || !isset($mtPeer['.id'])) {
            $mt->disconnect();
            $bot->sendMessage($chatId, "❌ Peer در میکروتیک یافت نشد.");
            return;
        }

        $result = $enable ? $mt->enablePeer($mtPeer['.id']) : $mt->disablePeer($mtPeer['.id']);
        $mt->disconnect();

        // Check for success or error in response
        $success = false;
        $errorMsg = '';
        foreach ($result as $item) {
            if ($item === '!done') {
                $success = true;
                break;
            }
            if ($item === '!trap') {
                foreach ($result as $r) {
                    if (is_array($r) && isset($r['message'])) {
                        $errorMsg = $r['message'];
                        break;
                    }
                }
                break;
            }
        }

        if ($success) {
            $status = $enable ? 'فعال' : 'غیرفعال';
            $emoji = $enable ? '✅' : '❌';
            $bot->sendMessage($chatId, "$emoji Peer <b>{$peer['name']}</b> با موفقیت $status شد.");
        } else {
            $errText = $errorMsg ? ": $errorMsg" : '';
            $bot->sendMessage($chatId, "❌ خطا در تغییر وضعیت peer$errText");
        }

    } catch (Exception $e) {
        $bot->sendMessage($chatId, "❌ خطا در ارتباط با میکروتیک: " . $e->getMessage());
    }
}

/**
 * Show complete peer information
 */
function showPeerInfo(int $chatId, string $name, TelegramBot $bot, WireGuardManager $wgManager, array $config): void
{
    $peer = $wgManager->getPeerByName($name);

    if (!$peer) {
        $bot->sendMessage($chatId, "❌ Peer با نام <b>$name</b> یافت نشد.");
        return;
    }

    // Calculate days since creation
    $createdDate = new DateTime($peer['created_at']);
    $now = new DateTime();
    $diff = $now->diff($createdDate);
    $daysUsed = $diff->days;

    $text = "ℹ️ <b>اطلاعات کامل: {$peer['name']}</b>\n\n";
    $text .= "🆔 <b>ID:</b> <code>{$peer['id']}</code>\n";
    $text .= "📛 <b>نام:</b> <code>{$peer['name']}</code>\n";
    $text .= "🌐 <b>IP:</b> <code>{$peer['ip']}</code>\n";
    $text .= "🔑 <b>Public Key:</b>\n<code>{$peer['public_key']}</code>\n";
    $text .= "📅 <b>تاریخ ایجاد:</b> {$peer['created_at']}\n";
    $text .= "⏱ <b>مدت استفاده:</b> $daysUsed روز\n";
    $text .= "🔄 <b>در میکروتیک:</b> " . ($peer['mikrotik_added'] ? '✅ بله' : '❌ خیر') . "\n";

    // Get traffic stats if in MikroTik
    if ($peer['mikrotik_added']) {
        try {
            $mt = new MikroTikAPI(
                $config['mikrotik']['host'],
                $config['mikrotik']['port'],
                $config['mikrotik']['username'],
                $config['mikrotik']['password']
            );

            $mt->connect();
            $mtPeer = $mt->findPeerByPublicKey($peer['public_key']);

            if ($mtPeer && isset($mtPeer['.id'])) {
                $traffic = $mt->getPeerTraffic($mtPeer['.id']);

                if ($traffic) {
                    $rxFormatted = formatBytes($traffic['rx_bytes']);
                    $txFormatted = formatBytes($traffic['tx_bytes']);
                    $totalFormatted = formatBytes($traffic['rx_bytes'] + $traffic['tx_bytes']);

                    $text .= "\n📊 <b>آمار مصرف:</b>\n";
                    $text .= "📥 دانلود: $txFormatted\n";
                    $text .= "📤 آپلود: $rxFormatted\n";
                    $text .= "📦 مجموع: $totalFormatted\n";

                    if ($traffic['last_handshake']) {
                        $text .= "🤝 آخرین handshake: {$traffic['last_handshake']}\n";
                    }

                    $status = $traffic['disabled'] ? '❌ غیرفعال' : '✅ فعال';
                    $text .= "🔌 وضعیت: $status";
                }
            }

            $mt->disconnect();

        } catch (Exception $e) {
            $text .= "\n⚠️ خطا در دریافت آمار از میکروتیک";
        }
    }

    $bot->sendMessage($chatId, $text);
}

/**
 * Format bytes to human readable format
 */
function formatBytes(int $bytes, int $precision = 2): string
{
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];

    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }

    return round($bytes, $precision) . ' ' . $units[$i];
}
