<?php
/**
 * Telegram Bot Handler
 * هندلر ربات تلگرام
 */

class TelegramBot
{
    private string $token;
    private array $adminIds;
    private string $apiUrl;
    
    public function __construct(string $token, array $adminIds = [])
    {
        $this->token = $token;
        $this->adminIds = $adminIds;
        $this->apiUrl = "https://api.telegram.org/bot{$token}/";
    }
    
    public function getUpdate(): ?array
    {
        $input = file_get_contents('php://input');
        return json_decode($input, true);
    }
    
    public function sendMessage(int $chatId, string $text, array $options = []): array
    {
        $params = array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $options);
        
        return $this->request('sendMessage', $params);
    }
    
    public function sendDocument(int $chatId, string $filePath, string $caption = ''): array
    {
        $params = [
            'chat_id' => $chatId,
            'caption' => $caption,
            'parse_mode' => 'HTML',
        ];
        
        return $this->uploadFile('sendDocument', $params, 'document', $filePath);
    }
    
    public function sendPhoto(int $chatId, string $filePath, string $caption = ''): array
    {
        $params = [
            'chat_id' => $chatId,
            'caption' => $caption,
            'parse_mode' => 'HTML',
        ];
        
        return $this->uploadFile('sendPhoto', $params, 'photo', $filePath);
    }
    
    public function sendMediaGroup(int $chatId, array $media): array
    {
        $params = [
            'chat_id' => $chatId,
            'media' => json_encode($media),
        ];
        
        return $this->request('sendMediaGroup', $params);
    }
    
    public function setWebhook(string $url): array
    {
        return $this->request('setWebhook', ['url' => $url]);
    }
    
    public function deleteWebhook(): array
    {
        return $this->request('deleteWebhook');
    }
    
    public function getWebhookInfo(): array
    {
        return $this->request('getWebhookInfo');
    }
    
    public function isAdmin(int $userId): bool
    {
        return in_array($userId, $this->adminIds);
    }
    
    public function sendKeyboard(int $chatId, string $text, array $buttons, bool $oneTime = true): array
    {
        $keyboard = [
            'keyboard' => $buttons,
            'one_time_keyboard' => $oneTime,
            'resize_keyboard' => true,
        ];
        
        return $this->sendMessage($chatId, $text, [
            'reply_markup' => json_encode($keyboard),
        ]);
    }
    
    public function sendInlineKeyboard(int $chatId, string $text, array $buttons): array
    {
        $keyboard = [
            'inline_keyboard' => $buttons,
        ];
        
        return $this->sendMessage($chatId, $text, [
            'reply_markup' => json_encode($keyboard),
        ]);
    }
    
    public function removeKeyboard(int $chatId, string $text): array
    {
        return $this->sendMessage($chatId, $text, [
            'reply_markup' => json_encode(['remove_keyboard' => true]),
        ]);
    }
    
    public function answerCallbackQuery(string $callbackId, string $text = '', bool $showAlert = false): array
    {
        return $this->request('answerCallbackQuery', [
            'callback_query_id' => $callbackId,
            'text' => $text,
            'show_alert' => $showAlert,
        ]);
    }
    
    public function editMessageText(int $chatId, int $messageId, string $text, array $options = []): array
    {
        $params = array_merge([
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $options);
        
        return $this->request('editMessageText', $params);
    }
    
    private function request(string $method, array $params = []): array
    {
        $ch = curl_init($this->apiUrl . $method);
        
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $params,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
        ]);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['ok' => false, 'error' => $error];
        }
        
        return json_decode($response, true) ?: ['ok' => false, 'error' => 'Invalid response'];
    }
    
    private function uploadFile(string $method, array $params, string $fileField, string $filePath): array
    {
        if (!file_exists($filePath)) {
            return ['ok' => false, 'error' => 'File not found: ' . $filePath];
        }
        
        $params[$fileField] = new CURLFile($filePath);
        
        $ch = curl_init($this->apiUrl . $method);
        
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $params,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_HTTPHEADER => ['Content-Type: multipart/form-data'],
        ]);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['ok' => false, 'error' => $error];
        }
        
        return json_decode($response, true) ?: ['ok' => false, 'error' => 'Invalid response'];
    }
}
