<?php
/**
 * MikroTik RouterOS API Client
 * کلاس ارتباط با API میکروتیک
 */

class MikroTikAPI
{
    private $socket;
    private $host;
    private $port;
    private $username;
    private $password;
    private $timeout = 5;
    private $connected = false;
    private $debug = false;
    
    public function __construct(string $host, int $port, string $username, string $password)
    {
        $this->host = $host;
        $this->port = $port;
        $this->username = $username;
        $this->password = $password;
    }
    
    public function connect(): bool
    {
        $this->socket = @fsockopen($this->host, $this->port, $errno, $errstr, $this->timeout);
        
        if (!$this->socket) {
            throw new Exception("Cannot connect to MikroTik: $errstr ($errno)");
        }
        
        // Login
        $response = $this->sendCommand('/login', [
            '=name=' . $this->username,
            '=password=' . $this->password,
        ]);
        
        if (isset($response[0]) && $response[0] === '!done') {
            $this->connected = true;
            return true;
        }
        
        throw new Exception("Login failed");
    }
    
    public function disconnect(): void
    {
        if ($this->socket) {
            fclose($this->socket);
            $this->socket = null;
            $this->connected = false;
        }
    }
    
    public function sendCommand(string $command, array $params = []): array
    {
        $this->writeWord($command);
        
        foreach ($params as $param) {
            $this->writeWord($param);
        }
        
        $this->writeWord('');
        
        return $this->readResponse();
    }
    
    private function writeWord(string $word): void
    {
        $len = strlen($word);
        
        if ($len < 0x80) {
            fwrite($this->socket, chr($len));
        } elseif ($len < 0x4000) {
            $len |= 0x8000;
            fwrite($this->socket, chr(($len >> 8) & 0xFF) . chr($len & 0xFF));
        } elseif ($len < 0x200000) {
            $len |= 0xC00000;
            fwrite($this->socket, chr(($len >> 16) & 0xFF) . chr(($len >> 8) & 0xFF) . chr($len & 0xFF));
        } elseif ($len < 0x10000000) {
            $len |= 0xE0000000;
            fwrite($this->socket, chr(($len >> 24) & 0xFF) . chr(($len >> 16) & 0xFF) . chr(($len >> 8) & 0xFF) . chr($len & 0xFF));
        } else {
            fwrite($this->socket, chr(0xF0) . chr(($len >> 24) & 0xFF) . chr(($len >> 16) & 0xFF) . chr(($len >> 8) & 0xFF) . chr($len & 0xFF));
        }
        
        fwrite($this->socket, $word);
    }
    
    private function readWord(): string
    {
        $byte = ord(fread($this->socket, 1));
        
        if ($byte & 0x80) {
            if (($byte & 0xC0) == 0x80) {
                $len = (($byte & ~0xC0) << 8) + ord(fread($this->socket, 1));
            } elseif (($byte & 0xE0) == 0xC0) {
                $len = (($byte & ~0xE0) << 16) + (ord(fread($this->socket, 1)) << 8) + ord(fread($this->socket, 1));
            } elseif (($byte & 0xF0) == 0xE0) {
                $len = (($byte & ~0xF0) << 24) + (ord(fread($this->socket, 1)) << 16) + (ord(fread($this->socket, 1)) << 8) + ord(fread($this->socket, 1));
            } elseif ($byte == 0xF0) {
                $len = (ord(fread($this->socket, 1)) << 24) + (ord(fread($this->socket, 1)) << 16) + (ord(fread($this->socket, 1)) << 8) + ord(fread($this->socket, 1));
            }
        } else {
            $len = $byte;
        }
        
        if ($len > 0) {
            return fread($this->socket, $len);
        }
        
        return '';
    }
    
    private function readResponse(): array
    {
        $response = [];
        $currentItem = [];
        
        while (true) {
            $word = $this->readWord();
            
            if ($word === '') {
                if (!empty($currentItem)) {
                    $response[] = $currentItem;
                    $currentItem = [];
                }
                continue;
            }
            
            if ($word === '!done' || $word === '!trap' || $word === '!fatal') {
                if (!empty($currentItem)) {
                    $response[] = $currentItem;
                }
                $response[] = $word;
                
                if ($word === '!done' || $word === '!fatal') {
                    break;
                }
            } elseif ($word === '!re') {
                if (!empty($currentItem)) {
                    $response[] = $currentItem;
                }
                $currentItem = [];
            } elseif (strpos($word, '=') === 0) {
                $parts = explode('=', substr($word, 1), 2);
                if (count($parts) == 2) {
                    $currentItem[$parts[0]] = $parts[1];
                }
            }
        }
        
        return $response;
    }
    
    // WireGuard specific methods
    
    public function addWireGuardPeer(string $interface, string $publicKey, string $allowedAddress, string $comment = '', string $rxRate = '', string $txRate = ''): array
    {
        $params = [
            '=interface=' . $interface,
            '=public-key=' . $publicKey,
            '=allowed-address=' . $allowedAddress,
        ];

        if (!empty($comment)) {
            $params[] = '=comment=' . $comment;
        }

        // Add rate limit (rx-rate = download speed for client, tx-rate = upload speed for client)
        if (!empty($rxRate)) {
            $params[] = '=rx-rate=' . $rxRate;
        }

        if (!empty($txRate)) {
            $params[] = '=tx-rate=' . $txRate;
        }

        return $this->sendCommand('/interface/wireguard/peers/add', $params);
    }
    
    public function removeWireGuardPeer(string $id): array
    {
        return $this->sendCommand('/interface/wireguard/peers/remove', [
            '=.id=' . $id,
        ]);
    }
    
    public function getWireGuardPeers(string $interface = ''): array
    {
        $params = [];
        if (!empty($interface)) {
            $params[] = '?interface=' . $interface;
        }
        
        $response = $this->sendCommand('/interface/wireguard/peers/print', $params);
        
        // Filter out !done and !trap
        return array_filter($response, fn($item) => is_array($item));
    }
    
    public function findPeerByComment(string $comment): ?array
    {
        $peers = $this->getWireGuardPeers();
        
        foreach ($peers as $peer) {
            if (isset($peer['comment']) && $peer['comment'] === $comment) {
                return $peer;
            }
        }
        
        return null;
    }
    
    public function findPeerByPublicKey(string $publicKey): ?array
    {
        $peers = $this->getWireGuardPeers();

        foreach ($peers as $peer) {
            if (isset($peer['public-key']) && $peer['public-key'] === $publicKey) {
                return $peer;
            }
        }

        return null;
    }

    /**
     * Enable a WireGuard peer
     */
    public function enablePeer(string $id): array
    {
        return $this->sendCommand('/interface/wireguard/peers/enable', [
            '=.id=' . $id,
        ]);
    }

    /**
     * Disable a WireGuard peer
     */
    public function disablePeer(string $id): array
    {
        return $this->sendCommand('/interface/wireguard/peers/disable', [
            '=.id=' . $id,
        ]);
    }

    /**
     * Get peer traffic statistics
     * Returns rx (received by server = uploaded by client) and tx (transmitted by server = downloaded by client)
     */
    public function getPeerTraffic(string $id): ?array
    {
        $peers = $this->getWireGuardPeers();

        foreach ($peers as $peer) {
            if (isset($peer['.id']) && $peer['.id'] === $id) {
                return [
                    'rx_bytes' => $peer['rx'] ?? 0,
                    'tx_bytes' => $peer['tx'] ?? 0,
                    'current_endpoint_address' => $peer['current-endpoint-address'] ?? null,
                    'current_endpoint_port' => $peer['current-endpoint-port'] ?? null,
                    'last_handshake' => $peer['last-handshake'] ?? null,
                    'disabled' => isset($peer['disabled']) && $peer['disabled'] === 'true',
                ];
            }
        }

        return null;
    }

    /**
     * Set rate limit for a peer
     */
    public function setPeerRateLimit(string $id, string $rxRate, string $txRate): array
    {
        $params = [
            '=.id=' . $id,
        ];

        if (!empty($rxRate)) {
            $params[] = '=rx-rate=' . $rxRate;
        }

        if (!empty($txRate)) {
            $params[] = '=tx-rate=' . $txRate;
        }

        return $this->sendCommand('/interface/wireguard/peers/set', $params);
    }
}
